<?php
declare (strict_types=1);

namespace App\Controller\Admin\API\User;

use App\Controller\Admin\Base;
use App\Entity\Query\Delete;
use App\Entity\Query\Get;
use App\Entity\Query\Save;
use App\Interceptor\Admin;
use App\Interceptor\PostDecrypt;
use App\Model\UserLevel as Model;
use App\Service\Common\Query;
use Hyperf\Database\Model\Builder;
use Hyperf\Database\Model\Relations\Relation;
use Kernel\Annotation\Inject;
use Kernel\Annotation\Interceptor;
use Kernel\Annotation\Validator;
use Kernel\Context\Interface\Response;
use Kernel\Exception\JSONException;
use Kernel\Exception\RuntimeException;
use Kernel\Waf\Filter;

#[Interceptor(class: [PostDecrypt::class, Admin::class], type: Interceptor::API)]
class Level extends Base
{
    #[Inject]
    private Query $query;


    /**
     * @return Response
     * @throws RuntimeException
     */
    public function get(): Response
    {
        $map = $this->request->post();
        $get = new Get(Model::class);
        $get->setWhere($map);
        $get->setOrderBy("sort", "asc");

        $data = $this->query->get($get, function (Builder $builder) use ($map) {
            if (isset($map['user_id']) && $map['user_id'] > 0) {
                $builder = $builder->where("user_id", $map['user_id']);
            } else {
                $builder = $builder->whereNull("user_id");
            }

            return $builder->with(["user" => function (Relation $relation) {
                $relation->select(["id", "username", "avatar"]);
            }])->withCount('member');
        });

        $data[0]['default'] = true;

        return $this->json(data: ['list' => $data]);
    }

    /**
     * @return Response
     * @throws JSONException
     */
    #[Validator([
        [\App\Validator\Admin\Level::class, "name"]
    ])]
    public function save(): Response
    {
        $map = $this->request->post();
        $privilegeIntroduce = $this->request->post("privilege_introduce", Filter::NORMAL);
        $privilegeContent = $this->request->post("privilege_content", Filter::NORMAL);
        $save = new Save(Model::class);
        $save->enableCreateTime();
        $save->setMap($map, ["icon", "name", "upgrade_price", "sort", "is_upgradable"]);
        $save->addForceMap("upgrade_requirements", json_encode($this->getUpgradeRequirements($map)));
        $privilegeIntroduce && $save->addForceMap("privilege_introduce", $privilegeIntroduce);
        $privilegeIntroduce && $save->addForceMap("privilege_content", $privilegeContent);
        if (isset($map['sort']) && $map['sort'] === "") {
            $level = Model::query()->whereNull("user_id")->orderBy("sort", "desc")->first();
            if ($level) {
                $save->addForceMap("sort", $level->sort + 1);
            }
        }
        try {
            $this->query->save($save);
        } catch (\Exception $exception) {
            throw new JSONException("保存失败，错误：" . $exception->getMessage());
        }
        return $this->response->json(message: "保存成功");
    }


    /**
     * @param array $map
     * @return array
     */
    private function getUpgradeRequirements(array $map): array
    {
        $requirements = [
            "total_consumption_amount" => 0,
            "total_recharge_amount" => 1,
            "total_referral_count" => 2,
            "total_profit_amount" => 3
        ];

        $data = [];
        if (isset($map['upgrade_requirements']) && is_array($map['upgrade_requirements'])) {
            foreach ($requirements as $key => $val) {
                if (in_array($val, $map['upgrade_requirements']) && isset($map[$key]) && $map[$key] > 0) {
                    $data[$key] = $map[$key];
                }
            }
        }
        return $data;
    }


    /**
     * @return Response
     * @throws JSONException
     */
    #[Validator([
        [\App\Validator\Common::class, "id"]
    ])]
    public function del(): Response
    {
        $id = $this->request->post("id", Filter::INTEGER);
        $level = Model::query()->find($id);

        if (!$level) {
            throw new JSONException("等级不存在");
        }


        if ($level->user_id > 0 ? Model::query()->where("user_id", $level->user_id)->count() <= 1 : Model::query()->whereNull("user_id")->count() <= 1) {
            throw new JSONException("必须保留1个默认等级");
        }

        //判断是否存在该会员等级
        if (\App\Model\User::query()->where("level_id", $id)->exists()) {
            throw new JSONException("该会员等级下有会员，无法删除");
        }

        $delete = new Delete(Model::class, [$id]);
        $this->query->delete($delete);
        return $this->response->json(message: "删除成功");
    }
}